/**
 * Copyright (c) 2020 Raspberry Pi (Trading) Ltd.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include <stdio.h>

#include "pico/stdlib.h"
#include "pico/time.h"
#include "hardware/clocks.h"
#include "hardware/vreg.h"
#define CHIPS_IMPL
#include "6502.c"
//#include "fake6502.h"
#include "6522.h"
#include <limits.h>


#define VIA_BASE_ADDRESS UINT16_C(0xA000)

// If this is active, then an overclock will be applied
#define OVERCLOCK

// Delay startup by so many seconds
#define START_DELAY 6

// The address at which to put your ROM file
#define ROM_START 0x8000
//Size of your ROM
#define ROM_SIZE 0x8000
// Your rom file in C array format
#define ROM_FILE "sym.h"
// Variable in which your rom data is stored
#define ROM_VAR sym_pico_bin


m6522_t via;
uint32_t gpio_dirs;
uint32_t gpio_outs;

uint8_t speed = 0;


//Raspberry Pi Pico GPIO Register is 32 bits
//xxxxxxxxxxxxxxxxBBBBBBBBAAAAAAAA  VIA
//                    ^       ^
//                   15-8    7-0    Pico
//Need to bit shift both the mask and data in order to access GPIO 15-8
#define GPIO_PORTA_MASK 0xFF  // PORTA of via is translated to GPIO pins 0 to 7
#define GPIO_PORTB_MASK 0xFF << 8  // PORTB of via is translated to GPIO pins 8 to 15
#define GPIO_PORTA_BASE_PIN 0  // data mask
#define GPIO_PORTB_BASE_PIN 8  //data mask


#include ROM_FILE
#define R_VAR ROM_VAR
#define R_START ROM_START
#define R_SIZE ROM_SIZE
uint32_t old_ticks = 0;

uint8_t ddra_value = 0;
uint8_t ddrb_value = 0;

uint8_t porta_value = 0;
uint8_t portb_value = 0;

uint8_t mem[0x10000];
absolute_time_t start;
bool running = true;

uint64_t via_pins = 0;

uint8_t access = 0;


//------------------
//  Read Rpi Ports
//-------------------

uint64_t read_ports() {

   uint64_t portreadA; 
   uint64_t portreadB;

    portreadA = gpio_get_all() & 0b0000000011111111;
    portreadB = gpio_get_all() & 0b1111111100000000;

    return portreadB  |= portreadA;    //returns  0bBBBBBBBBAAAAAAAA
      
}

//------------------
//  Write Rpi Ports
//-------------------


//----------------
//   VIA Update
//-----------------

void via_update() {

    // Update RPi Ports
    uint16_t getregister;
    uint16_t getregisterB;

    getregister = read6502(0xA000) << 8 | read6502(0xA001);
    gpio_put_masked(0b1111111111111111, getregister);

    getregister = read6502(0xA002) <<8 | read6502(0xA003);
    gpio_set_dir_masked(0b1111111111111111, getregister);

    //Call 6502 IRQ routine if 6522 has asserted it.  (I have now moved this to the Callback Function)
    // if ((uint32_t)(via_pins & 0XFFFFFFFF) & (uint32_t)(M6522_IRQ & 0XFFFFFFFF)) {
    //     printf("via irq\n");
    //     irq6502(); 
    // }
}


//----------
//   READ
//----------
uint8_t read6502(uint16_t address) {

    if (address == 0xf004) {                       //INCHR 
        int16_t ch = getchar_timeout_us(100);
        if (ch == PICO_ERROR_TIMEOUT) {
           return 0;
        }
        return (uint8_t) ch & 0xFF;


    } else if ((address & 0xFFF0) == VIA_BASE_ADDRESS) {
        
        via_pins = 0;
        via_pins &= ~(M6522_RS_PINS | M6522_CS2);                                 // CS2 low
        via_pins |= (M6522_RW | M6522_CS1 | ((uint16_t)M6522_RS_PINS & address)); // RW high (Read) CS1 High

        // Read the physical RPi ports, then OR this to the above (RW/CS) control lines for sending to 6522tick (6522.h emulator)

        uint64_t temp;
        temp = read_ports();         //returns obBBBBBBBBAAAAAAAA
        temp = temp << 48;           //shift into correct position as required by the 6522tick 64 bit string
        via_pins = via_pins |= temp; //OR with rest of the control line data


        via_pins = m6522_tick(&via, via_pins); //send the 64 bit pin state, then receive 64 bits of data back from the VIA (6522.h)
           
        uint8_t vdata = M6522_GET_DATA(via_pins);  //rotate 16 bits right from MSB from previous m6522_tick() which contains the register result

        return vdata;    //return the 6522 register contents received back from 6522tick 

    }

    return mem[address];   //Not INCHR or VIA, so just READ and return a general memory location.  
}




//----------
//  Write
//----------

void write6502(uint16_t address, uint8_t value) {  

    if (address == 0xf001) {     //OUTCHR
        printf("%c", value);


    } else if ((address & 0xFFF0) == VIA_BASE_ADDRESS) {   
        
        //printf("writing to VIA %04X val: %02X\n", address, value);

        via_pins = 0;

        via_pins &= ~(M6522_RW | M6522_RS_PINS | M6522_CS2);            // RW low (write) CS2 low
        
        via_pins |= (M6522_CS1 | ((uint16_t)M6522_RS_PINS & address));  // CS1 High   

        M6522_SET_DATA(via_pins, value);  //shifts data (value) into correct position within 64 bit 'via_pins' variable

        //Send the 64 bit config via 6522tick call to the 6522.h emulated register
        via_pins = m6522_tick(&via, via_pins);

        via_update();
        

    } else if (address < 0x8000){       //check to make sure we are not clobbering Monitor, RAE and BAS. This prevents BASIC crashing the system when determining memory
        
        mem[address] = value;           //WRITE 0000 - 7FFF block memory address  
    
    } else if (address >= 0x9000  && address <= 0x9FFF){

        mem[address] = value;           //WRITE 9000 - 9FFF block memory address. Allowed writes at this time to facilitate debugging.

    } else if (address >= 0xF000  && address <= 0xFFFF){

        mem[address] = value;           //WRITE F000 - FFFF block memory address

    } else if (address == 0xAC01){      //Simulate port that controls memory write protect
                                                              
        mem[address] = value;  

    } else if (address == 0xA402){      //Beeper port
                                                              
        mem[address] = value;  
        
        if ((value & 0x06) == 0x06){
        gpio_put(18,true);
        //printf("beep");
        }
            else {
            gpio_put(18,false);

     }

    } else if (address >= 0xA600  && address <= 0xA67F){      //SYSTEM RAM
        
        access = mem[0xAC01];                                 //check if System RAM write protect bit set
        access = access & 1;

        if (access == 1){                                                     
        mem[address] = value;                                  //WRITE system RAM 
    
        }  
      

    } else  {
    }

}



//-------------------------------------------------------------
//                  Hooked Callback
// This routine called by 6502 after execution of instruction
// which then performs a m6522tick to keep 6522 in step.
// HAVE NOW REMOVED THE 6522TICK AS IT WAS SLOWING DOWN EMULATOR 
//-------------------------------------------------------------

void callback() {
 
    // one tick for each clock to keep accurate time
    for (uint16_t i = 0; i < clockticks6502 - old_ticks; i++) {

        //via_pins = m6522_tick(&via, via_pins);    //removing this resolves the problems with the 1541DOS

        if ((uint32_t)(via_pins & 0XFFFFFFFF) & (uint32_t)(M6522_IRQ & 0XFFFFFFFF)) {
        printf("via irq\n");
        irq6502(); 
    }
    }

    //via_update();  //updates RPi ports, which will also result in ticking the 6522 //SLOWING DOWN TOO MUCH
   
    old_ticks = clockticks6502;

    // Attempt to measure speed with CRO connected to GPIO. 
    // Arrive here approx every 100 - 800uS. 
    // Pulse width approx 10nS which suggests minimal overhead in GPIO SDK
        gpio_put(19,1);
        gpio_put(19,0);
        
    

    
}


//------------------
//     STARTUP
//------------------
int main() {
#ifdef OVERCLOCK
    vreg_set_voltage(VREG_VOLTAGE_1_15);
    set_sys_clock_khz(280000, true);  //280000
#endif
    stdio_init_all();

    for(uint8_t i = START_DELAY; i > 0; i--) {
        printf("Starting in %d \n", i);
        sleep_ms(1000);
    }

    printf("Starting\n");

    if (R_START + R_SIZE > 0x10000) {
        printf("Your rom will not fit. Either adjust ROM_START or ROM_SIZE\n");
        while(1) {}
    }
    for (int i = R_START; i < R_SIZE + R_START; i++) {
        mem[i] = R_VAR[i-R_START];
    }

    hookexternal(callback);    //this sets up regular calls to the Callback function which will (perform a 6522 tick) << have removed call
    reset6502();

    //Initialise GPIO
    gpio_init(0);
    gpio_init(1);
    gpio_init(2);
    gpio_init(3);
    gpio_init(4);
    gpio_init(5);
    gpio_init(6);
    gpio_init(7);
    gpio_init(8);
    gpio_init(9);
    gpio_init(10);
    gpio_init(11);
    gpio_init(12);
    gpio_init(13);
    gpio_init(14);
    gpio_init(15);

    
    

#ifdef VIA_BASE_ADDRESS
    
// setup VIA
    m6522_init(&via);
    m6522_reset(&via);
    
    gpio_dirs = 0; //GPIO_PORTB_MASK | GPIO_PORTA_MASK;
    gpio_outs = 0;
    
    // Init GPIO
    gpio_init_mask(gpio_dirs);
    gpio_set_dir_all_bits(gpio_dirs);

    gpio_set_dir(16,0);  //create soft reset pin

    gpio_init(18);       //Beeper
    gpio_set_dir(18,GPIO_OUT);

    gpio_init(19);       //Pulse out to facilitate timing measurement within Callback routine
    gpio_set_dir(19,GPIO_OUT); 

 gpio_set_dir(17,GPIO_OUT);
    mem[0xAC01] = 1;  //initially allow writes to System Ram

#endif



//------------------
//    MAIN LOOP
//------------------
    start = get_absolute_time();

    while (running) {
        step6502();
        //exec6502(3000);

        if (gpio_get(16)==1){    //button pressed?
            reset6502();         //soft reset GPIO16
        }
    }
    return 0;
}
