; Source listing of Elektor´s Junior Computer Version D
; Written by Alois Nachtmann, 7. February 1980
; Highly minimized Version, 11. November 2025 UJ
;------------------------------------------------------------------------------

VERSION                = 1.0
;------------------------------------------------------------------------------

; Memory locations of the 6532
RIOT_BASE              = $1780
RIOT_RAM               = $1A00
RIOT_PORTS             = $1A80
PAD                    = RIOT_PORTS            ; Data register port A
PADD                   = RIOT_PORTS + 1        ; Data direction register port A
PBD                    = RIOT_PORTS + 2        ; Data register port B
PBDD                   = RIOT_PORTS + 3        ; Data direction register port B

; Temporary variables in zero page
TEMP                   = $FF                   ; Temporary Byte
MODE                   = $FE                   ; Input mode (0 = DA MODE, !0 = AD MODE)
KEY                    = $FD                   ; Key-Code
ADDRH                  = $FC                   ; Execution Address High
ADDRL                  = $FB                   ; Execution Address Low
INH                    = $FA                   ; Input data
DBYTES                 = $F8                   ; Number of Display bytes
ACC                    = $F7                   ; Accumulator
XREG                   = $F6                   ; Index X-Register
YREG                   = $F5                   ; Index Y-Register 
SPTR                   = $F4                   ; Stack Pointer
PSTAT                  = $F3                   ; Status Register
PCH                    = $F2                   ; Program Counter High
PCL                    = $F1                   ; Program Counter Low

; Interrupt vectors
NMIL                   = $1A7A                 ; NMI lower byte
NMIH                   = $1A7B                 ; NMI higher byte
IRQL                   = $1A7E                 ; IRQ lower byte
IRQH                   = $1A7F                 ; IRQ higher byte

KEYAD                  = $10                   ; AD-key value
KEYDA                  = $11                   ; DA-key value
KEYSTEP                = $12                   ; PLUS-key value
KEYGO                  = $13                   ; GO-key value
KEYPC                  = $14                   ; PC-key value
KEYILL                 = $15                   ; Illegal key value
;------------------------------------------------------------------------------

* = $FC00                                      ; Compile for this address

RESET                  lda #$02                ; Pre-set address $0200 at reset
                       ldy #$00
                       sta ADDRH
                       sty ADDRL
                       sty TTY_START_ADDR      ; Clear monitor start address flag

NOADRST                lda #$1E                ; No address reset, PB1---PB4 are outputs
                       sta PBDD
                       lda #$04                ; SET P-register
                       sta PSTAT
                       lda #$00                ; $00 = DA-mode, $03 = AD-mode
                       sta MODE                ; Set DA-mode
                       sta DBYTES              ; Display ADDRH, ADDRL, INH
                       ldx #$FF                ; Initialize the stack pointer
                       txs
                       stx SPTR
                       lda TTY_START_ADDR      ; Check if TTY load has set a start address
                       bne NOADDR              ; Do not set the standart if so
NOADDR                 cld
                       sei

START                  jsr SCAND               ; Display data specified by ADDRH and ADDRL
                       bne START               ; Wait until key is released

STARA                  jsr SCAND               ; Display data specified by point
                       beq STARA               ; Any key pressed?
                       jsr SCAND               ; Debounce key
                       beq STARA               ; Any key still pressed?
                       jsr GETKEY              ; Yes, decode key and return key in accu

GOEXEC                 cmp #KEYGO              ; GO-key pressed?
                       bne ADMODE
                       ldx SPTR                ; Get current Stack Pointer
                       txs
                       lda ADDRH               ; Start execution at ADDRH and ADDRL
                       pha
                       lda ADDRL
                       pha
                       lda PSTAT               ; Restore current P register
                       pha
                       ldx XREG
                       ldy YREG
                       lda ACC
                       rti                     ; Execute program

ADMODE                 cmp #KEYAD              ; AD-key pressed?
                       bne DAMODE
                       lda #$03                ; Set AD-mode
                       sta MODE
                       bne STEPA

DAMODE                 cmp #KEYDA              ; DA-key pressed?
                       bne STEP
                       lda #$00                ; Set DA-mode
                       sta MODE
                       beq STEPA

STEP                   cmp #KEYSTEP            ; PLUS-key pressed?
                       bne PCKEY
                       inc ADDRL
                       bne STEPA
                       inc ADDRH
STEPA                  jmp START

PCKEY                  cmp #KEYPC              ; PC-key pressed?
                       bne ILLKEY
                       jsr TTY_START_HEX       ; Initialize the serial interface
                       jmp STEPA

ILLKEY                 cmp #KEYILL             ; Illegal key?
                       bpl STEPA               ; Ignore if so

DATA                   sta KEY                 ; Save key
                       ldy MODE                ; Y=0 is data mode, else address mode
                       bne ADDRESS
                       lda (ADDRL),y           ; Get specified data by point
                       asl A
                       asl A                   ; Shift low order nibble to high order
                       asl A
                       asl A
                       ora KEY                 ; Data with key
                       sta (ADDRL),y           ; Restore data
                       jmp STEPA

ADDRESS                ldx #$04                ; 4 shifts
ADLOOP                 asl ADDRL               ; ADDRH, ADDRL 4 positions to left
                       ROL ADDRH
                       dex
                       bne ADLOOP
                       lda ADDRL
                       ora KEY                 ; Restore address
                       sta ADDRL
                       jmp STEPA
;------------------------------------------------------------------------------

; SCANDS shows the contents of the display buffer
; AK scans the keyboard, returns with A=0 if no key is pressed
; and with A <> 0 if a key is pressed when SCAND OR SCANDS are left, PA0...PA7 are inputs
SCAND                  ldy #$00
                       lda (ADDRL),y           ; Get data specified by ADDRL
                       sta INH

SCANDS                 lda #$7F
                       sta PADD                ; PA0...PA6 IS output
                       ldx #$08                ; Enable display
                       ldy DBYTES              ; Get DBYTES

SCDSA                  lda ADDRH               ; Output 1st byte
                       jsr SHOW 
                       dey
                       beq SCDSB               ; More DBYTES?
                       lda ADDRL
                       jsr SHOW                ; Output 2nd byte if so
                       dey
                       beq SCDSB               ; More DBYTES?
                       lda INH
                       jsr SHOW                ; Output 3rd byte if so

SCDSB                  lda #$00
                       sta PADD                ; PA0...PA7 are inputsINPUT
AK                     ldy #$03                ; Scan 3 rows
                       ldx #$00                ; Reset row counter 

ONEKEY                 lda #$FF

AKA                    stx PBD                 ; Output row number
                       inx                     ; Enable following row
                       inx
                       and PAD                 ; Input row pattern
                       dey                     ; ALL ROWS SCANNED?
                       bne AKA
                       ldy #$06                ; TURN DISPLAY OFF
                       sty PBD
                       ora #$80                ; Set bit7 = 1
                       eor #$FF                ; Invert key pattern
                       rts
;------------------------------------------------------------------------------

; SHOW displays the buffer contents. The X register is used as a scan counter
SHOW                   pha                     ; Save display
                       sty TEMP                ; Save Y register
                       lsr a
                       lsr a                   ; Get high order nibble
                       lsr a
                       lsr a
                       jsr CONVD               ; Output high order nibble
                       pla                     ; Get displayY again
                       and #$0F                ; Mask off high order nibble
                       jsr CONVD               ; Output low order nibble
                       ldy TEMP                ; Restore Y register
                       rts
;------------------------------------------------------------------------------

; CONVD controls the display 
CONVD                  tay                     ; Use nibble as index
                       lda SEGMENT_LUT,y       ; Get the segment pattern
                       sta PAD                 ; Output the segment pattern
                       stx PBD                 ; Enable the digit 
                       ldy #$7f                ; Dela Cycles ($7F)

DELAY                  dey                     ; Delay about 500µs
                       bpl DELAY
                       sty PAD                 ; Turn Segments off
                       ldy #$06
                       sty PBD                 ; Turn Display off
                       inx                     ; Enable next digit
                       inx
                       rts
;------------------------------------------------------------------------------

; GETKEY converts a key to a hex number
GETKEY                 ldx #$21                ; Start at row 0

GETKEA                 ldy #$01                ; Get one row
                       jsr ONEKEY              ; A=0, no key is pressed
                       bne KEYIN
                       cpx #$27
                       bne GETKEA              ; All rows scanned?
                       lda #$15                ; Return if the key is invalid
                       rts
;------------------------------------------------------------------------------

KEYIN                  ldy #$FF

KEYINA                 asl A                   ; Shift left until Y = key number
                       bcs KEYINB
                       iny
                       bpl KEYINA

KEYINB                 txa
                       and #$0F                ; Mask out MSN
                       lsr A                   ; Divide by 2
                       tax
                       tya
                       bpl KEYIND

KEYINC                 clc
                       adc #$07                ; Add row offset

KEYIND                 dex
                       bne KEYINC
                       rts
;------------------------------------------------------------------------------

NMI                    jmp (NMIL)              ; Jump to a user selectable NMI vector
IRQ                    jmp (IRQL)              ; Jump to a user selectable IRQ vector
;------------------------------------------------------------------------------

.include "junior_hex_loader.asm"               ; Include the MOS-HEX Loader source file

; The lookup table converts a hex number to a 7-segment pattern
SEGMENT_LUT            .byte $40,$79,$24,$30,$19,$12,$02,$78 ; '01234567'
                       .byte $00,$10,$08,$03,$46,$21,$06,$0E ; '89ABCDEF'

* = $FFFA                                      ; Vectors at the end of the memory

                       .word NMI               ; NMI vector
                       .word RESET             ; RESET vector
                       .word IRQ               ; IRQ or BRK vector
