; Hex-Loader (Intel-Hex and MOS-Hex), TTY Rx/Tx 2400 Baud, November 2025 UJ
;------------------------------------------------------------------------------

PB0 = TX, PB5 = RX
;------------------------------------------------------------------------------

; MOS-Hex Format (also called Paper-Tape-Format)
; RECORD MARK    ";" Colon 0x3B
; RECLEN         Data Bytes
; LOAD OFFSET    16-Bit-Address (Big-Endian)
; INFO or DATA   (RECLEN x 2 Zeichen)
; CHKSUM         16-Bit (LSB is used)

; Intel-Hex Format
; RECORD MARK    ":" Colon 0x3A
; RECLEN         Data Bytes
; LOAD OFFSET    16-Bit-Address (Big-Endian)
; RECTYP         (00..05)
; INFO or DATA   (RECLEN x 2 Zeichen)
; CHKSUM         8-Bit
;------------------------------------------------------------------------------

; Timer-Addresses
RIOT_TMR_RD            = RIOT_BASE + 4         ; Read without flag clear
RIOT_TMR_RDCLR         = RIOT_BASE + 5         ; Read with flag clear (Bit7 = UF)
RIOT_TMR_W1            = RIOT_BASE + 4         ; Write presc /1
RIOT_TMR_W8            = RIOT_BASE + 5         ; Write presc /8
RIOT_TMR_W64           = RIOT_BASE + 6         ; Write presc /64
RIOT_TMR_W1024         = RIOT_BASE + 7         ; Write presc /1024
;------------------------------------------------------------------------------

TTY_TMP_PUTC           = $EF                   ; Used by TTY_PUTC
TTY_TMP                = $EE                   ; Used in TTY_PUTDEC
TTY_TMP_GETC           = $ED                   ; Used by TTY_GETC
TTY_TMP_GETHEX         = $EC                   ; Used by TTY_GETHEX
TTY_LINE_BYTES         = $EB                   ; Number of line bytes LB
TTY_PTR_HI             = $EA                   ; Store address high
TTY_PTR_LO             = $E9                   ; Store address low
TTY_START_ADDR         = $E8                   ; Flag to store monitor start address 
TTY_CS                 = $E7                   ; 8-bit checksum
TTY_REC_CNT_HI         = $E6                   ; High byte of received bytes counter
TTY_REC_CNT_LO         = $E5                   ; Low byte of received bytes counter
TTY_TMPY               = $E4                   ; Temp Y used in TTY_PUTS
TTY_PSTR_HI            = $E3                   ; String pointer high byte
TTY_PSTR_LO            = $E2                   ; String pointer low byte
TTY_FORMAT             = $E1                   ; Hex-Format 0 = MOS-Hex (PTF), 1 = Intel-Hex
;------------------------------------------------------------------------------

TTY_TX_PRT_DIR         = PBDD
TTY_TX_PRT             = PBD
TTY_TXBIT              = %00000001             ; PB0 = TX

TTY_RX_PRT_DIR         = PBDD
TTY_RX_PRT             = PBD
TTY_RXBIT              = %00100000             ; PB5 = RX
TTY_RXBIT_INV          = %11011111             ; Inverse of TTY_RXBIT

BIT_DELAY              = 71                    ; 2400 Bd @ 1 MHz
HALF_DELAY             = 36
;------------------------------------------------------------------------------

TTY_INIT               cld                     ; Clear Decimal Flag
                       lda TTY_TX_PRT_DIR
                       ora #TTY_TXBIT
                       sta TTY_TX_PRT_DIR      ; TX is Output
                       lda TTY_TX_PRT
                       ora #TTY_TXBIT
                       lda TTY_TX_PRT          ; TX idle high

                       lda TTY_RX_PRT_DIR      ; RX is input
                       and #TTY_RXBIT_INV
                       sta TTY_RX_PRT_DIR

                       lda #<STR_TTY           ; Display message "TTY>"
                       sta TTY_PSTR_LO
                       lda #>STR_TTY
                       sta TTY_PSTR_HI
                       jsr TTY_PUTS
                       rts
;------------------------------------------------------------------------------

TTY_START_HEX          jsr TTY_INIT            ; Initialize the TTY
                       lda #<STR_RECEIVING     ; Display "receiving..."
                       sta TTY_PSTR_LO
                       lda #>STR_RECEIVING
                       sta TTY_PSTR_HI
                       jsr TTY_PUTS
                       lda #$00 
                       sta TTY_START_ADDR      ; Clear flag for monitor start address
                       sta TTY_REC_CNT_HI      ; Clear high byte of received bytes counter
                       sta TTY_REC_CNT_LO      ; Clear low byte of received bytes counter
                       jsr TTY_LOAD_LINE       ; Call the loader
                       jmp NOADRST             ; Jump to monitor reset, keep the loader address
;------------------------------------------------------------------------------

TTY_GO                 lda ADDRH
                       ldy ADDRL
                       tax
                       tya
                       sec
                       sbc #1
                       pha
                       txa
                       sbc #0
                       pha
                       rts
;------------------------------------------------------------------------------

TTY_DEL_BIT            pha
                       txa
                       pha
                       ldx #BIT_DELAY

DB                     dex
                       bne DB
                       pla
                       tax
                       pla
                       rts
;------------------------------------------------------------------------------

TTY_DEL_HBIT           pha
                       txa
                       pha
                       ldx #HALF_DELAY

DHB                    dex
                       bne DHB
                       pla
                       tax
                       pla
                       rts
;------------------------------------------------------------------------------

TTY_PUTC               pha
                       sta TTY_TMP_PUTC
                       lda #0
                       sta TTY_RX_PRT          ; Startbit = 0
                       jsr TTY_DEL_BIT
                       ldy #8

L_TXB                  lda TTY_TMP_PUTC
                       and #1
                       beq L_TX0
                       lda #TTY_TXBIT
                       sta TTY_RX_PRT          ; Bit = 1
                       bne L_TXN

L_TX0                  lda #0
                       sta PBD                 ; Bit = 0

L_TXN                  lsr TTY_TMP_PUTC
                       jsr TTY_DEL_BIT
                       dey
                       bne L_TXB
                       lda #TTY_TXBIT
                       sta TTY_RX_PRT          ; Stopbit = 1
                       jsr TTY_DEL_BIT
                       pla
                       rts
;------------------------------------------------------------------------------

TTY_PUTS               ldy #0

PS_LOOP                lda (TTY_PSTR_LO),y
                       beq PS_END
                       sty TTY_TMPY
                       jsr TTY_PUTC
                       ldy TTY_TMPY
                       iny
                       bne PS_LOOP

PS_END                 rts
;------------------------------------------------------------------------------

TTY_PUTDEC             ldx #$00                ; Count hundreds
PD_100                 cmp #$64                ; 100 dec
                       bcc PD_10
                       sbc #$64
                       inx
                       bne PD_100

PD_10                  stx TTY_TMP             ; Hundreds -> TTY_TMP
                       ldx #$00                ; Count tens
PD_10L                 cmp #$0A                ; 10 dec
                       bcc PD_OUT
                       sbc #$0A
                       inx
                       bne PD_10L

PD_OUT                 tay                     ; Ones to Y
                       sty TTY_TMPY            ; Save Y

                       lda TTY_TMP             ; Display hundreds?
                       beq PD_SKIP_HUND
                       clc
                       adc #'0'
                       jsr TTY_PUTC
PD_SKIP_HUND
                       lda TTY_TMP             ; Display tens?
                       bne PD_PRINT_TEN        ; H > 0?
                       cpx #$00
                       beq PD_SKIP_TEN
PD_PRINT_TEN
                       txa
                       clc
                       adc #'0'
                       jsr TTY_PUTC
PD_SKIP_TEN
                       ldy TTY_TMPY            ; Restore Y 
                       tya
                       clc
                       adc #'0'
                       jsr TTY_PUTC
                       rts
;------------------------------------------------------------------------------

TTY_PUTHEX             pha
                       lsr a
                       lsr a
                       lsr a
                       lsr a                   ; High-Nibble
                       tax
                       lda HEX_LUT,x
                       jsr TTY_PUTC
                       pla
                       and #$0F                ; Low-Nibble
                       tax
                       lda HEX_LUT,x
                       jsr TTY_PUTC
                       rts

HEX_LUT                .text "0123456789ABCDEF"; Lookup table for hex-characters
;------------------------------------------------------------------------------

TTY_CRLF               lda #$0D
                       jsr TTY_PUTC
                       lda #$0A
                       jsr TTY_PUTC
                       rts
;------------------------------------------------------------------------------

TTY_GETC               lda TTY_RX_PRT
                       and #TTY_RXBIT          ; Mask out the receive bit
                       bne TTY_GETC            ; Wait until line is low (startbit)
                       jsr TTY_DEL_HBIT        ; Delay until middle of startbit
                       ldy #8                  ; Load Y = bit counter
                       lda #$00                
                       sta TTY_TMP_GETC        ; Clear result byte

GC_RECB:               jsr TTY_DEL_BIT         ; Delay one bit time (middle of data bit)
                       lda TTY_RX_PRT
                       and #TTY_RXBIT          ; Mask out the receive bit
                       beq GC_BIT_LOW
                       sec                     ; Preset carry flag
                       bcs GC_BIT_HIGH         ; Leave it set if bit is hight
GC_BIT_LOW             clc                     ; else clear the carry flag

GC_BIT_HIGH            ror TTY_TMP_GETC        ; Carry flag in bit7, shift right, LSB first
                       dey                     ; Decrement the bit counter
                       bne GC_RECB             ; Receive next bit
                       jsr TTY_DEL_BIT         ; Delay one bit time (stopbit)
                       lda TTY_TMP_GETC
                       rts
;------------------------------------------------------------------------------

TTY_NIB                cmp #'0'                ; < '0' ?
                       bcc NIB_BAD
                       cmp #$3A                ; '9' + 1 = ':'
                       bcc NIB_OK
                       cmp #$41                ; 'A'
                       bcc NIB_BAD
                       cmp #$47                ; 'F' + 1
                       bcs NIB_BAD
                       sec
                       sbc #$41                ; 'A'
                       clc
                       adc #$0A                ; 10
                       sec
                       rts

NIB_OK                 sec
                       sbc #$30                ; '0'
                       sec
                       rts

NIB_BAD                clc
                       rts
;------------------------------------------------------------------------------

TTY_GETHEX             jsr TTY_GETC
                       jsr TTY_NIB
                       bcc GH_ERR
                       asl a 
                       asl a
                       asl a
                       asl a
                       sta TTY_TMP_GETHEX
                       jsr TTY_GETC
                       jsr TTY_NIB
                       bcc GH_ERR
                       ora TTY_TMP_GETHEX
                       sec
                       rts

GH_ERR                 clc
                       rts
;------------------------------------------------------------------------------

TTY_UPD_CS             clc
                       pha
                       adc TTY_CS              ; Add checksum to the accu
                       sta TTY_CS              ; And store it
                       pla
                       rts
;------------------------------------------------------------------------------

TTY_LOAD_LINE          lda #$00
                       sta TTY_FORMAT          ; Pre-set the code for 0 = MOS-Hex (Paper-Tape-Format)
                       jsr TTY_GETC            ; Get serial byte
                       cmp #';'                ; Check for semicolon ';' MOS-Hex, Paper-Tape-Format
                       beq LD_CLR_CS           ; If so, then branch to LD_CLR_CS
                       cmp #':'                ; Check for colon ';' Indexl-Hex
                       bne TTY_LOAD_LINE       ; No, wait for next character
                       inc TTY_FORMAT          ; Store the code for 1 = Intel-Hex Format

LD_CLR_CS              lda #$00
                       sta TTY_CS              ; Clear the checksum
                       jsr TTY_GETHEX          ; Get the number of bytes to read
                       php
                       sta TTY_LINE_BYTES      ; Save the number of line bytes
                       jsr TTY_UPD_CS          ; Update checksum
                       plp
                       beq LD_EXIT             ; Exit if bytes to read = 0
                       jsr TTY_GETHEX          ; Get Addr High Byte
                       sta TTY_PTR_HI          ; Store addr high
                       jsr TTY_UPD_CS          ; Update checksum
                       jsr TTY_GETHEX          ; Get Addr Low Byte
                       sta TTY_PTR_LO          ; Store addr low
                       jsr TTY_UPD_CS          ; Update checksum
                       lda TTY_FORMAT          ; Get the actual format
                       beq LD_SET_ADDR         ; Is 0 = MOS-Hex, branch to LD_SET_ADDR
                       jsr TTY_GETHEX          ; Get the record type 
                       jsr TTY_UPD_CS          ; Update checksum

LD_SET_ADDR            lda TTY_START_ADDR      ; Read monitor address set flag
                       bne LD_DATA             ; Monitor address already set?
                       lda TTY_PTR_HI
                       sta ADDRH               ; Store monitor GO ADDRH if it is the first record
                       lda TTY_PTR_LO
                       sta ADDRL               ; Stor monitor GO ADDRL if it is the first record
                       lda #$01
                       sta TTY_START_ADDR      ; Set monitor start address flag

LD_DATA                ldy #$00                ; Clear Y for storing data bytes

LD_LOOP                jsr TTY_GETHEX
                       sta (TTY_PTR_LO),y
                       jsr TTY_UPD_CS          ; Update checksum
                       inc TTY_PTR_LO
                       bne LD_NEXT
                       inc TTY_PTR_HI

LD_NEXT                inc TTY_REC_CNT_LO
                       bne LD_NOINC
                       inc TTY_REC_CNT_HI

LD_NOINC               dec TTY_LINE_BYTES      ; Data bytes -1
                       bne LD_LOOP             ; Loop if not zero

LD_CHK_MODE            lda TTY_FORMAT
                       beq LD_MOS 
                       jsr TTY_GETHEX          ; Get the transmitted 8-bit Intel-hex checksum    
                       clc
                       adc TTY_CS
                       bne LD_ERR              ; Error if transmitted checksum und calculated checksum are not equal
                       jmp TTY_LOAD_LINE       ; Get next line

LD_MOS                 jsr TTY_GETHEX          ; Get the transmitted MSB MOS-hex checksum    
                       jsr TTY_GETHEX          ; Get the transmitted LSB MOS-hex checksum
                       cmp TTY_CS
                       bne LD_ERR              ; Error if transmitted checksum und calculated checksum are not equal
                       jmp TTY_LOAD_LINE       ; Get next line

LD_EXIT                lda #<STR_OK            ; Display message "Ok, $"
                       sta TTY_PSTR_LO
                       lda #>STR_OK
                       sta TTY_PSTR_HI
                       jsr TTY_PUTS
                       lda TTY_REC_CNT_HI      ; Display the received bytes counter
                       jsr TTY_PUTHEX
                       lda TTY_REC_CNT_LO
                       jsr TTY_PUTHEX
                       lda #<STR_RECEIVED      ; Display "bytes received!"
                       sta TTY_PSTR_LO
                       lda #>STR_RECEIVED
                       sta TTY_PSTR_HI
                       jsr TTY_PUTS

                       lda TTY_FORMAT
                       beq LD_DISP_MOS
                       lda #<STR_INTEL         ; Display "Intel"
                       sta TTY_PSTR_LO
                       lda #>STR_MOS
                       sta TTY_PSTR_HI
                       jsr TTY_PUTS
                       jmp LD_DISP_HEX

LD_DISP_MOS            lda #<STR_MOS           ; Display "MOS"
                       sta TTY_PSTR_LO
                       lda #>STR_MOS
                       sta TTY_PSTR_HI
                       jsr TTY_PUTS

LD_DISP_HEX            lda #<STR_HEX           ; Display "-Hex"
                       sta TTY_PSTR_LO
                       lda #>STR_HEX
                       sta TTY_PSTR_HI
                       jsr TTY_PUTS
                       rts

LD_ERR                 lda #<STR_ERROR         ; Display message "Err!"
                       sta TTY_PSTR_LO
                       lda #>STR_ERROR
                       sta TTY_PSTR_HI
                       jsr TTY_PUTS
                       rts
;------------------------------------------------------------------------------

STR_TTY                .byte 13, 10
                       .text "TTY>"
                       .byte 13, 10, 0

STR_RECEIVING          .text "receiving..."
                       .byte 13, 10, 0

STR_OK                 .byte 13, 10
                       .text "Ok, $"
                       .byte 0

STR_RECEIVED           .text " bytes received!"
                       .byte 13, 10, 0

STR_MOS                .text "MOS"
                       .byte 0

STR_INTEL              .text "Intel"
                       .byte 0

STR_HEX                .text "-Hex"
                       .byte 13, 10, 0

STR_ERROR              .byte 13, 10
                       .text "Err!"
                       .byte 13, 10, 0
